import bpy
from mathutils import Vector
from math import atan2, radians

# Parameters
chair_loc = Vector((-0.75366, 2.6694, 1.5))  # Height 1.5 meters
desk_loc = Vector((-1.54352, -3.32974, 1.5))

frame_start = 1
frame_rot_end = X
frame_move_start = frame_rot_end + 1
frame_move_end = frame_move_start + X 
frame_end_rot_start = frame_move_end + 1
frame_end_rot_end = frame_end_rot_start + X

# Clear existing cameras
for obj in bpy.data.objects:
    if obj.type == 'CAMERA':
        bpy.data.objects.remove(obj, do_unlink=True)

# Add new camera
bpy.ops.object.camera_add(location=chair_loc)
cam = bpy.context.object
cam.name = 'ScanCam'

# === Insert only the following section here ===
cam.data.lens = 18         # Wider and larger field of view
cam.data.clip_start = 0.1  # Minimum visible distance
cam.data.clip_end = 10.0   # Maximum visible distance (shortened for interior focus)

# Calculate Z-axis rotation from start to end
delta = desk_loc - chair_loc
# Camera points along -Y axis, note the sign
z_rot_base = atan2(delta.x, delta.y)

#### 1. Rotate in place at start position ####
for f in range(frame_start, frame_rot_end + 1):
    frac = (f - frame_start) / (frame_rot_end - frame_start)
    z_rot = -z_rot_base + radians(360) * frac  # + for CCW, - for CW
    cam.location = chair_loc
    cam.rotation_euler = (radians(90), 0, z_rot)
    cam.keyframe_insert(data_path='location', frame=f)
    cam.keyframe_insert(data_path='rotation_euler', frame=f)

#### 2. Translate along a straight line ####
for i in range(frame_move_end - frame_move_start + 1):
    t = i / (frame_move_end - frame_move_start)
    pos = chair_loc.lerp(desk_loc, t)
    # Face desk_loc during movement
    z_rot = -z_rot_base
    cam.location = pos
    cam.rotation_euler = (radians(90), 0, z_rot)
    cam.keyframe_insert(data_path='location', frame=frame_move_start + i)
    cam.keyframe_insert(data_path='rotation_euler', frame=frame_move_start + i)

#### 3. Rotate in place at end position ####
for f in range(frame_end_rot_start, frame_end_rot_end + 1):
    frac = (f - frame_end_rot_start) / (frame_end_rot_end - frame_end_rot_start)
    z_rot = -z_rot_base + radians(360) * frac
    cam.location = desk_loc
    cam.rotation_euler = (radians(90), 0, z_rot)
    cam.keyframe_insert(data_path='location', frame=f)
    cam.keyframe_insert(data_path='rotation_euler', frame=f)

#### 4. Render settings ####
scene = bpy.context.scene
scene.frame_start = frame_start
scene.frame_end = frame_end_rot_end
scene.render.image_settings.file_format = 'FFMPEG'
scene.render.filepath = "scan_video.mp4"

